import { PrismaClient } from "@prisma/client";
import bcrypt from "bcryptjs";

const prisma = new PrismaClient();

async function hashPassword(password: string) {
  const salt = await bcrypt.genSalt(12);
  return bcrypt.hash(password, salt);
}

async function main() {
  const consumerEmail = "user@lefa.local";
  const merchantEmail = "merchant@lefa.local";
  const clearingEmail = "clearing@lefa.local";
  const adminEmail = "admin@lefa.local";

  const consumerPass = await hashPassword("password123");
  const merchantPass = await hashPassword("password123");
  const clearingPass = await hashPassword("password123");
  const adminPass = await hashPassword("password123");

  await prisma.user.upsert({
    where: { email: consumerEmail },
    update: {},
    create: {
      email: consumerEmail,
      passwordHash: consumerPass,
      fullName: "Demo User",
      phone: "+26770000001",
      role: "CONSUMER",
      kycLevel: "BASIC",
      wallet: { create: { balance: 50000n, currency: "BWP" } },
      kycProfile: {
        create: {
          status: "APPROVED",
          phoneVerifiedAt: new Date(),
          nationalId: "000000000",
          documentType: "OMANG",
          documentNumber: "000000000",
          submittedAt: new Date(),
          reviewedAt: new Date(),
        },
      },
    },
  });

  await prisma.user.upsert({
    where: { email: merchantEmail },
    update: {},
    create: {
      email: merchantEmail,
      passwordHash: merchantPass,
      fullName: "Demo Merchant",
      phone: "+26770000002",
      role: "MERCHANT",
      kycLevel: "BASIC",
      wallet: { create: { balance: 0n, currency: "BWP" } },
      merchant: {
        create: {
          legalName: "LEFA Demo Store (Pty) Ltd",
          tradingName: "LEFA Demo Store",
          category: "Retail",
          city: "Gaborone",
        },
      },
      kycProfile: {
        create: {
          status: "APPROVED",
          phoneVerifiedAt: new Date(),
          documentType: "OMANG",
          documentNumber: "000000001",
          submittedAt: new Date(),
          reviewedAt: new Date(),
        },
      },
    },
  });

  const merchantUser = await prisma.user.findUnique({ where: { email: merchantEmail } });
  const merchantProfile = merchantUser
    ? await prisma.merchantProfile.findUnique({ where: { userId: merchantUser.id } })
    : null;
  if (merchantProfile) {
    const existingLoc = await prisma.merchantLocation.findFirst({ where: { merchantId: merchantProfile.id } });
    if (!existingLoc) {
      await prisma.merchantLocation.create({
        data: {
          merchantId: merchantProfile.id,
          name: "Main Location",
          city: "Gaborone",
          address: "CBD",
        },
      });
    }
  }

  const clearing = await prisma.user.upsert({
    where: { email: clearingEmail },
    update: {},
    create: {
      email: clearingEmail,
      passwordHash: clearingPass,
      fullName: "LEFA Clearing (System)",
      role: "ADMIN",
      kycLevel: "FULL",
      wallet: { create: { balance: 0n, currency: "BWP" } },
      kycProfile: { create: { status: "APPROVED", reviewedAt: new Date(), submittedAt: new Date() } },
    },
  });

  await prisma.user.upsert({
    where: { email: adminEmail },
    update: {},
    create: {
      email: adminEmail,
      passwordHash: adminPass,
      fullName: "LEFA Admin",
      role: "ADMIN",
      kycLevel: "FULL",
      wallet: { create: { balance: 0n, currency: "BWP" } },
      kycProfile: { create: { status: "APPROVED", reviewedAt: new Date(), submittedAt: new Date() } },
    },
  });

  console.log("Seeded demo accounts:");
  console.log("- user@lefa.local / password123");
  console.log("- merchant@lefa.local / password123");
  console.log("- admin@lefa.local / password123");
  console.log(`- clearing@lefa.local / password123 (system)`);
  console.log(`Set env LEFA_CLEARING_USER_ID=${clearing.id} (recommended for production deployments)`);
}

main()
  .catch((e) => {
    console.error(e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
