import crypto from "crypto";
import { prisma } from "@/lib/prisma";

export type WebhookEventType =
  | "payment_intent.succeeded"
  | "payment_request.paid";

function sign(secret: string, payload: string) {
  return crypto.createHmac("sha256", secret).update(payload).digest("hex");
}

async function postWebhook(url: string, secret: string, event: any) {
  const body = JSON.stringify(event);
  const signature = sign(secret, body);

  const controller = new AbortController();
  const timeout = setTimeout(() => controller.abort(), 8000);
  try {
    await fetch(url, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "X-Lefa-Signature": signature,
        "X-Lefa-Event": event.type,
      },
      body,
      signal: controller.signal,
    });
  } finally {
    clearTimeout(timeout);
  }
}

export async function dispatchUserWebhooks(userId: string, type: WebhookEventType, data: any) {
  const endpoints = await prisma.webhookEndpoint.findMany({
    where: { userId, disabledAt: null },
  });
  if (endpoints.length === 0) return;

  const evt = {
    id: crypto.randomUUID(),
    type,
    createdAt: new Date().toISOString(),
    data,
  };

  await Promise.all(
    endpoints
      .filter((e) => {
        const allowed = (e.events || "").split(",").map((s) => s.trim()).filter(Boolean);
        return allowed.length === 0 || allowed.includes(type);
      })
      .map((e) => postWebhook(e.url, e.secret, evt).catch(() => null))
  );
}
