import { prisma } from "@/lib/prisma";
import { assertSpendingAllowed } from "@/lib/risk";

async function resolveClearingUserId(tx: typeof prisma) {
  const envId = process.env.LEFA_CLEARING_USER_ID;
  if (envId) return envId;
  const u = await (tx as any).user.findUnique({ where: { email: "clearing@lefa.local" } });
  if (u?.id) return u.id as string;
  throw new Error("MISSING_LEFA_CLEARING_USER_ID");
}

export async function ensureWallet(userId: string) {
  const existing = await prisma.walletAccount.findUnique({ where: { userId } });
  if (existing) return existing;
  return prisma.walletAccount.create({ data: { userId, balance: 0n, currency: "BWP" } });
}

export async function transferLefaToLefa(params: {
  fromUserId: string;
  toEmail: string;
  amountMinor: bigint;
  description?: string;
}) {
  const { fromUserId, toEmail, amountMinor, description } = params;
  if (amountMinor <= 0n) throw new Error("INVALID_AMOUNT");

  const toUser = await prisma.user.findUnique({ where: { email: toEmail.toLowerCase().trim() } });
  if (!toUser) throw new Error("RECIPIENT_NOT_FOUND");
  if (toUser.id === fromUserId) throw new Error("CANNOT_SEND_TO_SELF");

  return prisma.$transaction(async (tx) => {
    // Risk & limits (KYC-tier based)
    await assertSpendingAllowed({ userId: fromUserId, amountMinor, txType: "TRANSFER" });

    const fromWallet = await tx.walletAccount.upsert({
      where: { userId: fromUserId },
      update: {},
      create: { userId: fromUserId, balance: 0n, currency: "BWP" },
    });

    const toWallet = await tx.walletAccount.upsert({
      where: { userId: toUser.id },
      update: {},
      create: { userId: toUser.id, balance: 0n, currency: "BWP" },
    });

    if (fromWallet.balance < amountMinor) throw new Error("INSUFFICIENT_FUNDS");

    await tx.walletAccount.update({
      where: { id: fromWallet.id },
      data: { balance: { decrement: amountMinor } },
    });

    await tx.walletAccount.update({
      where: { id: toWallet.id },
      data: { balance: { increment: amountMinor } },
    });

    const t = await tx.transaction.create({
      data: {
        type: "TRANSFER",
        status: "POSTED",
        amount: amountMinor,
        currency: "BWP",
        description,
        fromUserId,
        toUserId: toUser.id,
      },
    });

    return { transaction: t, toUser: { id: toUser.id, email: toUser.email } };
  });
}

export async function transferToUserId(params: {
  fromUserId: string;
  toUserId: string;
  amountMinor: bigint;
  description?: string;
  txType?: "TRANSFER" | "PAYMENT";
  reference?: string;
}) {
  const { fromUserId, toUserId, amountMinor, description, txType = "PAYMENT", reference } = params;
  if (amountMinor <= 0n) throw new Error("INVALID_AMOUNT");
  if (fromUserId === toUserId) throw new Error("CANNOT_SEND_TO_SELF");

  return prisma.$transaction(async (tx) => {
    await assertSpendingAllowed({ userId: fromUserId, amountMinor, txType: txType === "TRANSFER" ? "TRANSFER" : "PAYMENT" });

    const fromWallet = await tx.walletAccount.upsert({
      where: { userId: fromUserId },
      update: {},
      create: { userId: fromUserId, balance: 0n, currency: "BWP" },
    });
    const toWallet = await tx.walletAccount.upsert({
      where: { userId: toUserId },
      update: {},
      create: { userId: toUserId, balance: 0n, currency: "BWP" },
    });
    if (fromWallet.balance < amountMinor) throw new Error("INSUFFICIENT_FUNDS");

    await tx.walletAccount.update({ where: { id: fromWallet.id }, data: { balance: { decrement: amountMinor } } });
    await tx.walletAccount.update({ where: { id: toWallet.id }, data: { balance: { increment: amountMinor } } });

    const t = await tx.transaction.create({
      data: {
        type: txType,
        status: "POSTED",
        amount: amountMinor,
        currency: "BWP",
        description,
        reference: reference ?? null,
        fromUserId,
        toUserId,
      },
    });

    return { transaction: t };
  });
}

export async function creditWallet(params: { userId: string; amountMinor: bigint; description?: string }) {
  const { userId, amountMinor, description } = params;
  if (amountMinor <= 0n) throw new Error("INVALID_AMOUNT");

  return prisma.$transaction(async (tx) => {
    const wallet = await tx.walletAccount.upsert({
      where: { userId },
      update: {},
      create: { userId, balance: 0n, currency: "BWP" },
    });

    await tx.walletAccount.update({
      where: { id: wallet.id },
      data: { balance: { increment: amountMinor } },
    });

    const t = await tx.transaction.create({
      data: {
        type: "TOPUP",
        status: "POSTED",
        amount: amountMinor,
        currency: "BWP",
        description: description ?? "Top-up",
        toUserId: userId,
      },
    });

    return { transaction: t };
  });
}

export async function debitWallet(params: { userId: string; amountMinor: bigint; description?: string }) {
  const { userId, amountMinor, description } = params;
  if (amountMinor <= 0n) throw new Error("INVALID_AMOUNT");

  return prisma.$transaction(async (tx) => {
    await assertSpendingAllowed({ userId, amountMinor, txType: "WITHDRAWAL" });

    const wallet = await tx.walletAccount.upsert({
      where: { userId },
      update: {},
      create: { userId, balance: 0n, currency: "BWP" },
    });
    if (wallet.balance < amountMinor) throw new Error("INSUFFICIENT_FUNDS");

    await tx.walletAccount.update({
      where: { id: wallet.id },
      data: { balance: { decrement: amountMinor } },
    });

    const t = await tx.transaction.create({
      data: {
        type: "WITHDRAWAL",
        status: "POSTED",
        amount: amountMinor,
        currency: "BWP",
        description: description ?? "Withdrawal",
        fromUserId: userId,
      },
    });
    return { transaction: t };
  });
}

/**
 * Used for guest checkout / external rails simulation.
 * In production this would be driven by real bank/wallet confirmations.
 */
export async function settleExternalPayment(params: {
  toUserId: string;
  amountMinor: bigint;
  description?: string;
  reference?: string;
}) {
  const { toUserId, amountMinor, description, reference } = params;
  if (amountMinor <= 0n) throw new Error("INVALID_AMOUNT");

  return prisma.$transaction(async (tx) => {
    const clearingUserId = await resolveClearingUserId(tx as any);
    if (clearingUserId === toUserId) throw new Error("INVALID_TO_USER");

    const clearingWallet = await tx.walletAccount.upsert({
      where: { userId: clearingUserId },
      update: {},
      create: { userId: clearingUserId, balance: 0n, currency: "BWP" },
    });

    const toWallet = await tx.walletAccount.upsert({
      where: { userId: toUserId },
      update: {},
      create: { userId: toUserId, balance: 0n, currency: "BWP" },
    });

    // Demo-friendly: always ensure clearing has enough float.
    if (clearingWallet.balance < amountMinor) {
      await tx.walletAccount.update({
        where: { id: clearingWallet.id },
        data: { balance: { increment: amountMinor * 1000n } },
      });
    }

    await tx.walletAccount.update({
      where: { id: clearingWallet.id },
      data: { balance: { decrement: amountMinor } },
    });

    await tx.walletAccount.update({
      where: { id: toWallet.id },
      data: { balance: { increment: amountMinor } },
    });

    const t = await tx.transaction.create({
      data: {
        type: "PAYMENT",
        status: "POSTED",
        amount: amountMinor,
        currency: "BWP",
        description,
        reference: reference ?? null,
        fromUserId: clearingUserId,
        toUserId,
      },
    });

    return { transaction: t };
  });
}
