"use client";

import { useState } from "react";
import { Button, Input } from "@/components/ui";

type Props = {
  user: {
    id: string;
    email: string;
    phone: string | null;
    kycLevel: string;
    kycStatus: string;
    phoneVerifiedAt: string | null;
  };
};

export function KycPanel({ user }: Props) {
  const [phone, setPhone] = useState(user.phone ?? "");
  const [challengeId, setChallengeId] = useState<string | null>(null);
  const [otp, setOtp] = useState("");
  const [devOtp, setDevOtp] = useState<string | undefined>();
  const [msg, setMsg] = useState<string | null>(null);
  const [docType, setDocType] = useState("OMANG");
  const [docNumber, setDocNumber] = useState("");

  async function requestOtp() {
    setMsg(null);
    const res = await fetch("/api/kyc/request-otp", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ phone }),
    });
    const j = await res.json().catch(() => ({}));
    if (!res.ok) return setMsg(j.error ?? "FAILED");
    setChallengeId(j.challengeId);
    setDevOtp(j.devOtp);
    setMsg("OTP sent (dev shows code). Enter it to verify your phone.");
  }

  async function verifyOtp() {
    setMsg(null);
    const res = await fetch("/api/kyc/verify-otp", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ challengeId, otp, phone }),
    });
    const j = await res.json().catch(() => ({}));
    if (!res.ok) return setMsg(j.error ?? "FAILED");
    setMsg("Phone verified. Reloading...");
    window.location.reload();
  }

  async function submitFull() {
    setMsg(null);
    const res = await fetch("/api/kyc/submit", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ documentType: docType, documentNumber: docNumber }),
    });
    const j = await res.json().catch(() => ({}));
    if (!res.ok) return setMsg(j.error ?? "FAILED");
    setMsg("Submitted for FULL KYC review. Admin can approve in /admin/kyc.");
    window.location.reload();
  }

  return (
    <div className="grid gap-4">
      <div className="rounded-2xl border border-zinc-200 bg-white p-4">
        <div className="text-sm font-medium">Current level</div>
        <div className="mt-1 text-2xl font-semibold">{user.kycLevel}</div>
        <div className="mt-1 text-xs text-zinc-600">Status: {user.kycStatus}</div>
      </div>

      <div className="rounded-2xl border border-zinc-200 bg-white p-4">
        <div className="text-sm font-medium">Step 1: Verify phone</div>
        <div className="mt-2 grid gap-2">
          <Input label="Phone" value={phone} onChange={(e) => setPhone(e.target.value)} placeholder="+2677xxxxxxx" />
          {user.phoneVerifiedAt ? (
            <div className="text-sm text-green-700">Phone verified ✓</div>
          ) : (
            <>
              {!challengeId ? (
                <Button onClick={requestOtp}>Send OTP</Button>
              ) : (
                <div className="grid gap-2">
                  <Input label="OTP" value={otp} onChange={(e) => setOtp(e.target.value)} placeholder="123456" />
                  {devOtp ? <div className="text-xs text-zinc-600">Dev OTP: <span className="font-mono">{devOtp}</span></div> : null}
                  <Button onClick={verifyOtp}>Verify</Button>
                </div>
              )}
            </>
          )}
        </div>
      </div>

      <div className="rounded-2xl border border-zinc-200 bg-white p-4">
        <div className="text-sm font-medium">Step 2: Upgrade to FULL (optional)</div>
        <div className="mt-2 grid gap-2">
          <label className="text-xs text-zinc-600">Document type</label>
          <select value={docType} onChange={(e) => setDocType(e.target.value)} className="rounded-xl border border-zinc-200 bg-white px-3 py-2 text-sm">
            <option value="OMANG">OMANG</option>
            <option value="PASSPORT">PASSPORT</option>
          </select>
          <Input label="Document number" value={docNumber} onChange={(e) => setDocNumber(e.target.value)} placeholder="e.g., 123456789" />
          <Button onClick={submitFull} disabled={!user.phoneVerifiedAt}>Submit FULL KYC</Button>
          {!user.phoneVerifiedAt ? <div className="text-xs text-zinc-600">Verify phone first to unlock FULL submission.</div> : null}
        </div>
      </div>

      {msg ? <div className="text-sm text-zinc-700">{msg}</div> : null}
    </div>
  );
}
