"use client";

import { useMemo, useState } from "react";
import { Button } from "@/components/ui";

type Props =
  | { mode: "PAY_REQUEST"; token: string }
  | { mode: "PAY_INTENT"; checkoutToken: string }
  | { mode: "TEXT_PAY"; token: string }
  | { mode: "POS"; token: string };

export function GuestPay(props: Props) {
  const [phone, setPhone] = useState("");
  const [challengeId, setChallengeId] = useState<string | null>(null);
  const [devOtp, setDevOtp] = useState<string | null>(null);
  const [code, setCode] = useState("");
  const [status, setStatus] = useState<string>("");
  const [busy, setBusy] = useState(false);

  const purpose = props.mode;

  const payBody = useMemo(() => {
    if (props.mode === "PAY_REQUEST") return { token: props.token };
    if (props.mode === "TEXT_PAY") return { token: props.token };
    if (props.mode === "POS") return { token: props.token };
    return { checkoutToken: props.checkoutToken };
  }, [props]);

  async function requestOtp() {
    setBusy(true);
    setStatus("");
    try {
      const res = await fetch("/api/guest/request-otp", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ phone, purpose }),
      });
      const json = await res.json();
      if (!res.ok) throw new Error(json?.error ?? "REQUEST_FAILED");
      setChallengeId(json.challengeId);
      setDevOtp(json.devOtp ?? null);
      setStatus("OTP sent.");
    } catch (e: any) {
      setStatus(e.message ?? "Failed to send OTP");
    } finally {
      setBusy(false);
    }
  }

  async function verifyOtp() {
    if (!challengeId) return;
    setBusy(true);
    setStatus("");
    try {
      const res = await fetch("/api/guest/verify-otp", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ phone, challengeId, code }),
      });
      const json = await res.json();
      if (!res.ok) throw new Error(json?.error ?? "VERIFY_FAILED");
      setStatus("Verified. You can pay now.");
    } catch (e: any) {
      setStatus(e.message ?? "Failed to verify OTP");
    } finally {
      setBusy(false);
    }
  }

  async function pay() {
    setBusy(true);
    setStatus("");
    try {
      const url =
        props.mode === "PAY_REQUEST"
          ? "/api/guest/pay-request"
          : props.mode === "PAY_INTENT"
          ? "/api/guest/pay-intent"
          : props.mode === "TEXT_PAY"
          ? "/api/guest/pay-text"
          : "/api/guest/pay-pos";
      const res = await fetch(url, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(payBody),
      });
      const json = await res.json();
      if (!res.ok) throw new Error(json?.error ?? "PAY_FAILED");
      setStatus("Payment completed.");
      // Refresh current route so status updates server-side
      window.location.reload();
    } catch (e: any) {
      setStatus(e.message ?? "Payment failed");
    } finally {
      setBusy(false);
    }
  }

  return (
    <div className="mt-4 rounded-xl border border-zinc-200 p-4">
      <div className="text-sm font-medium">Pay as guest</div>
      <div className="mt-1 text-xs text-zinc-600">Use OTP verification. (In dev, the OTP will display here.)</div>

      <div className="mt-3 grid gap-2">
        <input
          value={phone}
          onChange={(e) => setPhone(e.target.value)}
          placeholder="Phone number (e.g. +2677xxxxxxx)"
          className="w-full rounded-xl border border-zinc-200 px-3 py-2 text-sm"
        />
        <Button onClick={requestOtp} disabled={busy || phone.trim().length < 7}>
          Send OTP
        </Button>

        {challengeId ? (
          <>
            <input
              value={code}
              onChange={(e) => setCode(e.target.value)}
              placeholder="Enter OTP"
              className="w-full rounded-xl border border-zinc-200 px-3 py-2 text-sm"
            />
            <Button onClick={verifyOtp} disabled={busy || code.trim().length < 4}>
              Verify OTP
            </Button>
            <Button onClick={pay} disabled={busy} className="w-full">
              Pay now
            </Button>
          </>
        ) : null}

        {devOtp ? <div className="text-xs text-amber-700">DEV OTP: {devOtp}</div> : null}
        {status ? <div className="text-xs text-zinc-700">{status}</div> : null}
      </div>
    </div>
  );
}
