"use client";

import { signIn } from "next-auth/react";
import { useRouter, useSearchParams } from "next/navigation";
import { FormEvent, useMemo, useState } from "react";
import { Button, Card, CardBody, CardHeader, Container, Input, Label } from "@/components/ui";

export default function SignInPage() {
  const router = useRouter();
  const params = useSearchParams();
  const callbackUrl = useMemo(() => params.get("callbackUrl") ?? "/app", [params]);
  const [error, setError] = useState<string | null>(null);
  const [loading, setLoading] = useState(false);
  const [mode, setMode] = useState<"phone" | "email">("phone");
  const [otpChallengeId, setOtpChallengeId] = useState<string | null>(null);
  const [devOtp, setDevOtp] = useState<string | null>(null);

  async function onSubmitEmail(e: FormEvent<HTMLFormElement>) {
    e.preventDefault();
    setLoading(true);
    setError(null);

    const form = new FormData(e.currentTarget);
    const email = String(form.get("email") ?? "");
    const password = String(form.get("password") ?? "");

    const res = await signIn("credentials", {
      email,
      password,
      redirect: false,
      callbackUrl,
    });

    setLoading(false);

    if (!res || res.error) {
      setError("Invalid email or password");
      return;
    }

    router.push(res.url ?? callbackUrl);
  }

  async function onRequestOtp(e: FormEvent<HTMLFormElement>) {
    e.preventDefault();
    setLoading(true);
    setError(null);
    setDevOtp(null);

    const form = new FormData(e.currentTarget);
    const phone = String(form.get("phone") ?? "").trim();
    if (!phone) {
      setLoading(false);
      setError("Enter your phone number");
      return;
    }

    const res = await fetch("/api/auth/request-otp", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ phone }),
    }).then((r) => r.json());

    setLoading(false);
    if (!res?.ok) {
      setError(res?.error === "RATE_LIMIT" ? "Please wait a minute and try again" : "Failed to send OTP");
      return;
    }
    setOtpChallengeId(res.challengeId);
    if (res.devOtp) setDevOtp(res.devOtp);
  }

  async function onVerifyOtp(e: FormEvent<HTMLFormElement>) {
    e.preventDefault();
    setLoading(true);
    setError(null);

    const form = new FormData(e.currentTarget);
    const phone = String(form.get("phone") ?? "").trim();
    const otp = String(form.get("otp") ?? "").trim();
    if (!phone || !otp || !otpChallengeId) {
      setLoading(false);
      setError("Enter phone and OTP");
      return;
    }

    const res = await signIn("credentials", {
      loginType: "phone",
      phone,
      otp,
      challengeId: otpChallengeId,
      redirect: false,
      callbackUrl,
    });

    setLoading(false);
    if (!res || res.error) {
      setError("Invalid OTP");
      return;
    }
    router.push(res.url ?? callbackUrl);
  }

  return (
    <>
      <div className="min-h-screen bg-zinc-50">
        <Container>
          <div className="mx-auto max-w-md">
            <Card>
              <CardHeader title="Sign in" subtitle="Access your LEFA wallet and dashboard" />
              <CardBody>
                <div className="mb-4 flex gap-2">
                  <button
                    type="button"
                    className={`rounded-full px-3 py-1 text-sm ${mode === "phone" ? "bg-black text-white" : "bg-zinc-100"}`}
                    onClick={() => {
                      setMode("phone");
                      setError(null);
                    }}
                  >
                    Phone (OTP)
                  </button>
                  <button
                    type="button"
                    className={`rounded-full px-3 py-1 text-sm ${mode === "email" ? "bg-black text-white" : "bg-zinc-100"}`}
                    onClick={() => {
                      setMode("email");
                      setError(null);
                    }}
                  >
                    Email
                  </button>
                </div>

                {mode === "phone" ? (
                  <div className="space-y-4">
                    {!otpChallengeId ? (
                      <form className="space-y-4" onSubmit={onRequestOtp}>
                        <div>
                          <Label>Phone</Label>
                          <Input name="phone" type="tel" placeholder="+267 71 532 423" />
                        </div>
                        {error ? <div className="text-sm text-red-600">{error}</div> : null}
                        <Button type="submit" disabled={loading} className="w-full">
                          {loading ? "Sending…" : "Send OTP"}
                        </Button>
                      </form>
                    ) : (
                      <form className="space-y-4" onSubmit={onVerifyOtp}>
                        <div>
                          <Label>Phone</Label>
                          <Input name="phone" type="tel" placeholder="+267 71 532 423" />
                        </div>
                        <div>
                          <Label>OTP</Label>
                          <Input name="otp" type="text" placeholder="123456" />
                          {devOtp ? (
                            <div className="mt-1 text-xs text-zinc-500">Dev OTP: {devOtp}</div>
                          ) : null}
                        </div>
                        {error ? <div className="text-sm text-red-600">{error}</div> : null}
                        <Button type="submit" disabled={loading} className="w-full">
                          {loading ? "Verifying…" : "Verify & Sign in"}
                        </Button>
                        <button
                          type="button"
                          className="w-full text-center text-xs text-zinc-600"
                          onClick={() => {
                            setOtpChallengeId(null);
                            setDevOtp(null);
                          }}
                        >
                          Change phone
                        </button>
                      </form>
                    )}
                  </div>
                ) : (
                  <form className="space-y-4" onSubmit={onSubmitEmail}>
                    <div>
                      <Label>Email</Label>
                      <Input name="email" type="email" placeholder="you@example.com" />
                    </div>
                    <div>
                      <Label>Password</Label>
                      <Input name="password" type="password" placeholder="••••••••" />
                    </div>
                    {error ? <div className="text-sm text-red-600">{error}</div> : null}
                    <Button type="submit" disabled={loading} className="w-full">
                      {loading ? "Signing in…" : "Sign in"}
                    </Button>
                  </form>
                )}
              </CardBody>
            </Card>
            <div className="mt-4 text-center text-xs text-zinc-500">
              Demo accounts are in the README.
            </div>
          </div>
        </Container>
      </div>
    </>
  );
}
