"use client";

import { useMemo, useState } from "react";
import { Button, Input, Label } from "@/components/ui";

type Created = {
  url: string;
  token: string;
};

export function QRClient({ baseUrl }: { baseUrl: string }) {
  const [amount, setAmount] = useState("50");
  const [reference, setReference] = useState("QR-DEMO");
  const [description, setDescription] = useState("");
  const [created, setCreated] = useState<Created | null>(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const qrUrl = useMemo(() => {
    const link = created?.url ?? "";
    if (!link) return "";
    return `/api/qr?text=${encodeURIComponent(link)}`;
  }, [created]);

  async function create() {
    setLoading(true);
    setError(null);
    try {
      const res = await fetch("/api/payment-requests", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          amount: Number(amount),
          currency: "BWP",
          reference,
          description,
        }),
      });
      const json = await res.json();
      if (!res.ok) throw new Error(json?.error ?? "FAILED");
      setCreated({ url: json.url, token: json.token });
    } catch (e: any) {
      setError(e?.message ?? "FAILED");
    } finally {
      setLoading(false);
    }
  }

  const shareText = created ? `Pay me with LEFA: ${created.url}` : "";

  async function shareWhatsApp() {
    if (!created) return;
    const wa = `https://wa.me/?text=${encodeURIComponent(shareText)}`;
    window.open(wa, "_blank");
  }

  async function shareNative() {
    if (!created) return;
    const nav: any = navigator;
    if (nav.share) {
      await nav.share({ title: "LEFA Pay Request", text: shareText, url: created.url });
    } else {
      await navigator.clipboard.writeText(created.url);
      alert("Link copied");
    }
  }

  return (
    <div>
      <div className="grid gap-3">
        <div>
          <Label>Amount (BWP)</Label>
          <Input value={amount} onChange={(e) => setAmount(e.target.value)} inputMode="decimal" />
        </div>
        <div>
          <Label>Reference</Label>
          <Input value={reference} onChange={(e) => setReference(e.target.value)} />
        </div>
        <div>
          <Label>Description (optional)</Label>
          <Input value={description} onChange={(e) => setDescription(e.target.value)} />
        </div>

        <Button type="button" onClick={create} disabled={loading}>
          {loading ? "Creating…" : "Create pay request"}
        </Button>

        {error ? <div className="text-sm text-red-700">{error}</div> : null}

        {created ? (
          <div className="rounded-xl border border-zinc-200 p-4">
            <div className="text-xs text-zinc-600">Short pay link</div>
            <div className="mt-1 break-all text-sm font-medium">{created.url}</div>

            <div className="mt-3 flex flex-wrap gap-2">
              <Button
                type="button"
                className="bg-zinc-700 hover:bg-zinc-600"
                onClick={async () => navigator.clipboard.writeText(created.url)}
              >
                Copy link
              </Button>
              <Button type="button" className="bg-green-700 hover:bg-green-600" onClick={shareWhatsApp}>
                WhatsApp
              </Button>
              <Button type="button" className="bg-zinc-900 hover:bg-zinc-800" onClick={shareNative}>
                Share
              </Button>
            </div>

            <div className="mt-4 flex items-center justify-center">
              <img alt="QR" className="h-56 w-56" src={qrUrl} />
            </div>
            <div className="mt-2 text-center text-xs text-zinc-500">Scan to pay (LEFA app/web)</div>
          </div>
        ) : null}
      </div>
    </div>
  );
}
