import { TopNav } from "@/components/nav";
import { Button, Card, CardBody, CardHeader, Container } from "@/components/ui";
import { requireSession } from "@/lib/session";
import { transferLefaToLefa } from "@/lib/ledger";
import { prisma } from "@/lib/prisma";

export default async function CheckoutRequestPage({
  searchParams,
}: {
  searchParams: { to?: string; amount?: string; ref?: string; desc?: string };
}) {
  const session = await requireSession();

  const toEmail = (searchParams.to ?? "").toLowerCase().trim();
  const amount = Number(searchParams.amount ?? "0");
  const ref = (searchParams.ref ?? "").slice(0, 64);
  const desc = (searchParams.desc ?? "").slice(0, 140);

  const toUser = toEmail ? await prisma.user.findUnique({ where: { email: toEmail } }) : null;

  async function pay() {
    "use server";
    const amountMinor = BigInt(Math.round(amount * 100));
    const res = await transferLefaToLefa({
      fromUserId: session.uid,
      toEmail,
      amountMinor,
      description: desc || `QR payment ${ref}`,
    });
    return res.transaction.id;
  }

  const valid = !!toUser && amount > 0;

  return (
    <>
      <TopNav />
      <Container>
        <div className="mx-auto max-w-xl">
          <Card>
            <CardHeader title="Pay request" subtitle={valid ? "Confirm payment" : "Invalid request"} />
            <CardBody>
              {!valid ? (
                <div className="text-sm text-zinc-700">This payment request is missing or invalid.</div>
              ) : (
                <div className="space-y-4">
                  <div className="rounded-xl border border-zinc-200 p-4">
                    <div className="text-xs text-zinc-600">Pay to</div>
                    <div className="text-sm font-medium">{toUser?.email}</div>
                    <div className="mt-2 text-xs text-zinc-600">Amount</div>
                    <div className="text-2xl font-semibold">P {amount.toFixed(2)}</div>
                    {ref ? <div className="mt-2 text-xs text-zinc-600">Reference: {ref}</div> : null}
                    {desc ? <div className="mt-1 text-xs text-zinc-600">{desc}</div> : null}
                  </div>

                  <form action={pay}>
                    <Button type="submit" className="w-full">Pay now</Button>
                  </form>
                </div>
              )}
            </CardBody>
          </Card>
        </div>
      </Container>
    </>
  );
}
