import { TopNav } from "@/components/nav";
import { Button, Card, CardBody, CardHeader, Container, Input, Label } from "@/components/ui";
import { requireSession } from "@/lib/session";
import { prisma } from "@/lib/prisma";
import { fromMinor } from "@/lib/money";
import { demoTopupAction, externalFundingAction } from "@/app/actions/wallet";

export default async function WalletHome() {
  const session = await requireSession();

  const user = await prisma.user.findUnique({
    where: { id: session.uid },
    include: { wallet: true },
  });

  const balance = user?.wallet?.balance ?? 0n;

  const recent = await prisma.transaction.findMany({
    where: {
      OR: [{ fromUserId: session.uid }, { toUserId: session.uid }],
    },
    orderBy: { createdAt: "desc" },
    take: 10,
  });

  return (
    <>
      <TopNav />
      <Container>
        <div className="grid gap-6 md:grid-cols-2">
          <Card>
            <CardHeader title="Wallet" subtitle={user?.email ?? ""} />
            <CardBody>
              <div className="text-sm text-zinc-600">Available balance</div>
              <div className="mt-1 text-3xl font-semibold">P {fromMinor(balance)}</div>

              <div className="mt-6 grid gap-4">
                <div className="rounded-xl border border-zinc-200 bg-zinc-50 p-4">
                  <div className="text-sm font-medium">Add / Withdraw funds</div>
                  <div className="mt-1 text-xs text-zinc-600">
                    Phone-first funding (Orange Money, Smega, My Zaka, Bank). Adapters are stubbed and ready for real API wiring.
                  </div>
                  <form action={externalFundingAction} className="mt-3 grid gap-3">
                    <div className="grid grid-cols-2 gap-3">
                      <div>
                        <Label>Direction</Label>
                        <select
                          name="direction"
                          className="mt-1 w-full rounded-xl border border-zinc-200 bg-white px-3 py-2 text-sm"
                          defaultValue="TOPUP"
                        >
                          <option value="TOPUP">Top up</option>
                          <option value="WITHDRAWAL">Withdraw</option>
                        </select>
                      </div>
                      <div>
                        <Label>Provider</Label>
                        <select
                          name="provider"
                          className="mt-1 w-full rounded-xl border border-zinc-200 bg-white px-3 py-2 text-sm"
                          defaultValue="ORANGE_MONEY"
                        >
                          <option value="ORANGE_MONEY">Orange Money</option>
                          <option value="SMEGA">Smega</option>
                          <option value="MY_ZAKA">My Zaka</option>
                          <option value="BANK">Bank</option>
                        </select>
                      </div>
                    </div>
                    <div className="grid grid-cols-2 gap-3">
                      <div>
                        <Label>Phone / Account</Label>
                        <Input name="phoneOrAccount" type="text" placeholder="+267 71 532 423" />
                      </div>
                      <div>
                        <Label>Amount (BWP)</Label>
                        <Input name="amount" type="number" placeholder="200" />
                      </div>
                    </div>
                    <Button type="submit">Continue</Button>
                  </form>
                </div>

                <div className="rounded-xl border border-zinc-200 bg-zinc-50 p-4">
                  <div className="text-sm font-medium">Demo top-up</div>
                  <div className="mt-1 text-xs text-zinc-600">For testing without providers.</div>
                  <form action={demoTopupAction} className="mt-3 flex items-end gap-3">
                    <div className="flex-1">
                      <Label>Amount (BWP)</Label>
                      <Input name="amount" type="number" placeholder="200" />
                    </div>
                    <Button type="submit">Top up</Button>
                  </form>
                </div>
              </div>
            </CardBody>
          </Card>

          <Card>
            <CardHeader title="Recent activity" subtitle="Last 10 transactions" />
            <CardBody>
              <div className="space-y-3">
                {recent.length === 0 ? (
                  <div className="text-sm text-zinc-600">No transactions yet.</div>
                ) : (
                  recent.map((t) => {
                    const incoming = t.toUserId === session.uid;
                    return (
                      <div
                        key={t.id}
                        className="flex items-center justify-between rounded-xl border border-zinc-200 px-3 py-2"
                      >
                        <div>
                          <div className="text-sm font-medium">{t.type}</div>
                          <div className="text-xs text-zinc-600">
                            {t.description ?? ""} · {new Date(t.createdAt).toLocaleString()}
                          </div>
                        </div>
                        <div className={incoming ? "text-sm font-semibold" : "text-sm font-semibold text-zinc-900"}>
                          {incoming ? "+" : "-"}P {fromMinor(t.amount)}
                        </div>
                      </div>
                    );
                  })
                )}
              </div>
            </CardBody>
          </Card>
        </div>
      </Container>
    </>
  );
}
