import { TopNav } from "@/components/nav";
import { Button, Card, CardBody, CardHeader, Container, Input, Textarea } from "@/components/ui";
import { requireSession } from "@/lib/session";
import { prisma } from "@/lib/prisma";
import { z } from "zod";

const Schema = z.object({
  transactionId: z.string().min(5).max(60),
  title: z.string().min(3).max(80),
  description: z.string().min(5).max(1000),
});

export default async function NewDisputePage() {
  await requireSession();

  async function create(_: any, formData: FormData) {
    "use server";
    const session = await requireSession();
    const parsed = Schema.safeParse({
      transactionId: formData.get("transactionId"),
      title: formData.get("title"),
      description: formData.get("description"),
    });
    if (!parsed.success) return { ok: false, error: "INVALID_INPUT" };
    await prisma.dispute.create({
      data: {
        userId: session.uid,
        transactionId: parsed.data.transactionId,
        title: parsed.data.title,
        description: parsed.data.description,
      },
    });
    return { ok: true };
  }

  return (
    <>
      <TopNav />
      <Container>
        <div className="mx-auto max-w-xl">
          <Card>
            <CardHeader title="Report an issue" subtitle="Dispute a transaction" />
            <CardBody>
              <form action={create} className="grid gap-3">
                <Input name="transactionId" label="Transaction ID" placeholder="Paste the Tx ID" />
                <Input name="title" label="Title" placeholder="e.g., Wrong amount / Paid but not delivered" />
                <Textarea name="description" label="Description" placeholder="Explain what happened" rows={5} />
                <Button type="submit">Submit dispute</Button>
              </form>
              <div className="mt-3 text-xs text-zinc-600">
                Tip: You can find transaction IDs in Wallet → Recent activity.
              </div>
            </CardBody>
          </Card>
        </div>
      </Container>
    </>
  );
}
