import { z } from "zod";
import { prisma } from "@/lib/prisma";
import { requireSession } from "@/lib/session";
import { randomToken } from "@/lib/crypto";

const BodySchema = z.object({
  amount: z.number().positive(), // major units
  currency: z.string().optional().default("BWP"),
  reference: z.string().optional().default("").transform((v) => v.slice(0, 64)),
  description: z.string().optional().default("").transform((v) => v.slice(0, 140)),
});

export async function POST(req: Request) {
  const session = await requireSession();

  const json = await req.json().catch(() => null);
  const parsed = BodySchema.safeParse(json);
  if (!parsed.success) return Response.json({ error: "INVALID_BODY" }, { status: 400 });

  const amountMinor = BigInt(Math.round(parsed.data.amount * 100));
  const token = randomToken(18);

  const pr = await prisma.paymentRequest.create({
    data: {
      toUserId: session.uid,
      amount: amountMinor,
      currency: parsed.data.currency,
      reference: parsed.data.reference || null,
      description: parsed.data.description || null,
      token,
      status: "ACTIVE",
    },
  });

  const base = process.env.NEXTAUTH_URL ?? "http://localhost:3000";
  const url = `${base}/pay/r/${pr.token}`;

  return Response.json({
    id: pr.id,
    token: pr.token,
    amount: Number(pr.amount),
    currency: pr.currency,
    reference: pr.reference,
    description: pr.description,
    url,
  });
}
