import { NextRequest } from "next/server";
import { z } from "zod";
import { authenticateApiKey } from "@/lib/apiKeyAuth";
import { prisma } from "@/lib/prisma";
import { randomToken } from "@/lib/crypto";

const BodySchema = z.object({
  amount: z.number().int().positive(), // minor units
  currency: z.string().default("BWP"),
  reference: z.string().min(1).max(64),
  description: z.string().max(140).optional(),
});

export async function POST(req: NextRequest) {
  const apiKey = await authenticateApiKey(req.headers.get("authorization"));
  if (!apiKey) return Response.json({ error: "UNAUTHORIZED" }, { status: 401 });

  const user = await prisma.user.findUnique({ where: { id: apiKey.userId }, include: { merchant: true } });
  if (!user?.merchant) return Response.json({ error: "NOT_A_MERCHANT" }, { status: 403 });

  const json = await req.json().catch(() => null);
  const parsed = BodySchema.safeParse(json);
  if (!parsed.success) return Response.json({ error: "INVALID_BODY" }, { status: 400 });

  const checkoutToken = randomToken(24);

  const pi = await prisma.paymentIntent.create({
    data: {
      merchantId: user.merchant.id,
      amount: BigInt(parsed.data.amount),
      currency: parsed.data.currency,
      reference: parsed.data.reference,
      description: parsed.data.description,
      status: "REQUIRES_PAYMENT",
      checkoutToken,
    },
  });

  const checkoutUrl = `${process.env.NEXTAUTH_URL ?? "http://localhost:3000"}/checkout/pi/${checkoutToken}`;

  return Response.json({
    id: pi.id,
    status: pi.status,
    amount: Number(pi.amount),
    currency: pi.currency,
    reference: pi.reference,
    checkoutUrl,
  });
}
