import { z } from "zod";
import { prisma } from "@/lib/prisma";
import { requireSession } from "@/lib/session";

const BodySchema = z.object({
  documentType: z.string().min(2).max(20),
  documentNumber: z.string().min(3).max(40),
});

export async function POST(req: Request) {
  const session = await requireSession();
  const json = await req.json().catch(() => null);
  const parsed = BodySchema.safeParse(json);
  if (!parsed.success) return Response.json({ error: "INVALID_BODY" }, { status: 400 });

  const current = await prisma.kycProfile.findUnique({ where: { userId: session.uid } });
  if (!current?.phoneVerifiedAt) return Response.json({ error: "PHONE_NOT_VERIFIED" }, { status: 400 });

  await prisma.kycProfile.upsert({
    where: { userId: session.uid },
    update: {
      status: "PENDING",
      documentType: parsed.data.documentType,
      documentNumber: parsed.data.documentNumber,
      submittedAt: new Date(),
    },
    create: {
      userId: session.uid,
      status: "PENDING",
      documentType: parsed.data.documentType,
      documentNumber: parsed.data.documentNumber,
      submittedAt: new Date(),
    },
  });

  return Response.json({ ok: true });
}
