import { z } from "zod";
import { prisma } from "@/lib/prisma";
import { requireSession } from "@/lib/session";

const BodySchema = z.object({
  phone: z.string().min(7).max(20),
});

function makeOtp() {
  return Math.floor(100000 + Math.random() * 900000).toString();
}

export async function POST(req: Request) {
  const session = await requireSession();
  const json = await req.json().catch(() => null);
  const parsed = BodySchema.safeParse(json);
  if (!parsed.success) return Response.json({ error: "INVALID_BODY" }, { status: 400 });

  const phone = parsed.data.phone.trim();
  // Throttle: 1 OTP / 60s per phone.
  const oneMinuteAgo = new Date(Date.now() - 60_000);
  const recent = await prisma.otpChallenge.findFirst({
    where: { phone, createdAt: { gte: oneMinuteAgo }, purpose: "KYC_PHONE" },
    orderBy: { createdAt: "desc" },
  });
  if (recent) return Response.json({ error: "RATE_LIMIT" }, { status: 429 });

  const code = makeOtp();
  const expiresAt = new Date(Date.now() + 5 * 60_000);
  const ch = await prisma.otpChallenge.create({ data: { phone, purpose: "KYC_PHONE", code, expiresAt } });

  // Store phone on user (lite KYC)
  await prisma.user.update({ where: { id: session.uid }, data: { phone } });

  const devOtp = process.env.NODE_ENV !== "production" ? code : undefined;
  return Response.json({ ok: true, challengeId: ch.id, expiresAt: ch.expiresAt, devOtp });
}
