import { z } from "zod";
import { prisma } from "@/lib/prisma";
import { getGuestSession } from "@/lib/guest";
import { settleExternalPayment } from "@/lib/ledger";

const BodySchema = z.object({
  token: z.string().min(1),
});

export async function POST(req: Request) {
  const guest = getGuestSession();
  if (!guest) return Response.json({ error: "GUEST_UNAUTHENTICATED" }, { status: 401 });

  const json = await req.json().catch(() => null);
  const parsed = BodySchema.safeParse(json);
  if (!parsed.success) return Response.json({ error: "INVALID_BODY" }, { status: 400 });

  const r = await prisma.textPayRequest.findUnique({
    where: { token: parsed.data.token },
    include: { fromUser: true },
  });
  if (!r) return Response.json({ error: "NOT_FOUND" }, { status: 404 });
  if (r.status === "PAID") return Response.json({ ok: true, status: r.status, txId: r.paidTxId });

  const res = await settleExternalPayment({
    toUserId: r.fromUserId,
    amountMinor: r.amount,
    description: r.description ?? `Text-to-pay ${r.reference ?? ""}`.trim(),
    reference: r.reference ?? null ?? undefined,
  });

  await prisma.textPayRequest.update({
    where: { id: r.id },
    data: { status: "PAID", paidTxId: res.transaction.id },
  });

  return Response.json({ ok: true, status: "PAID", txId: res.transaction.id });
}
