import { z } from "zod";
import { prisma } from "@/lib/prisma";
import { getGuestSession } from "@/lib/guest";
import { settleExternalPayment } from "@/lib/ledger";
import { dispatchUserWebhooks } from "@/lib/webhooks";

const BodySchema = z.object({
  token: z.string().min(1),
});

export async function POST(req: Request) {
  const guest = getGuestSession();
  if (!guest) return Response.json({ error: "GUEST_UNAUTHENTICATED" }, { status: 401 });

  const json = await req.json().catch(() => null);
  const parsed = BodySchema.safeParse(json);
  if (!parsed.success) return Response.json({ error: "INVALID_BODY" }, { status: 400 });

  const pr = await prisma.paymentRequest.findUnique({
    where: { token: parsed.data.token },
    include: { toUser: true },
  });
  if (!pr) return Response.json({ error: "NOT_FOUND" }, { status: 404 });
  if (pr.status !== "ACTIVE") return Response.json({ ok: true, status: pr.status, txId: pr.paidTxId });

  const guestRow = await prisma.guestPayer.findUnique({ where: { id: guest.guestId } });
  if (!guestRow) return Response.json({ error: "GUEST_NOT_FOUND" }, { status: 401 });

  const ref = pr.reference ? `PR-${pr.reference}` : `PR-${pr.id}`;
  const res = await settleExternalPayment({
    toUserId: pr.toUserId,
    amountMinor: pr.amount,
    description: pr.description ?? `Guest QR payment ${pr.reference ?? ""}`.trim(),
    reference: ref,
  });

  const updated = await prisma.paymentRequest.update({
    where: { id: pr.id },
    data: {
      status: "PAID",
      paidTxId: res.transaction.id,
      paidByGuestId: guestRow.id,
    },
  });

  // Notify receiver webhooks (optional feature).
  await dispatchUserWebhooks(pr.toUserId, "payment_request.paid", {
    token: pr.token,
    amount: pr.amount.toString(),
    currency: pr.currency,
    reference: pr.reference,
    txId: res.transaction.id,
    guest: { id: guestRow.id, phone: guestRow.phone },
  });

  return Response.json({ ok: true, status: updated.status, txId: res.transaction.id });
}
