import { z } from "zod";
import { prisma } from "@/lib/prisma";
import { getGuestSession } from "@/lib/guest";
import { settleExternalPayment } from "@/lib/ledger";
import { dispatchUserWebhooks } from "@/lib/webhooks";

const BodySchema = z.object({
  token: z.string().min(1),
});

export async function POST(req: Request) {
  const guest = getGuestSession();
  if (!guest) return Response.json({ error: "GUEST_UNAUTHENTICATED" }, { status: 401 });

  const json = await req.json().catch(() => null);
  const parsed = BodySchema.safeParse(json);
  if (!parsed.success) return Response.json({ error: "INVALID_BODY" }, { status: 400 });

  const charge = await prisma.inStoreCharge.findUnique({ where: { token: parsed.data.token } });
  if (!charge) return Response.json({ error: "NOT_FOUND" }, { status: 404 });
  if (charge.status === "PAID") return Response.json({ ok: true, status: charge.status, txId: charge.paidTxId });

  const merchant = await prisma.merchantProfile.findUnique({ where: { id: charge.merchantId } });
  if (!merchant) return Response.json({ error: "MERCHANT_NOT_FOUND" }, { status: 404 });

  const guestRow = await prisma.guestPayer.findUnique({ where: { id: guest.guestId } });
  if (!guestRow) return Response.json({ error: "GUEST_NOT_FOUND" }, { status: 401 });

  const ref = `POS-${charge.id}`;
  const res = await settleExternalPayment({
    toUserId: merchant.userId,
    amountMinor: charge.amount,
    description: charge.description ?? `Guest POS payment${charge.reference ? ` ${charge.reference}` : ""}`,
    reference: ref,
  });

  const updated = await prisma.inStoreCharge.update({
    where: { id: charge.id },
    data: { status: "PAID", paidTxId: res.transaction.id },
  });

  await dispatchUserWebhooks(merchant.userId, "pos_charge.paid", {
    token: updated.token,
    amount: updated.amount.toString(),
    currency: updated.currency,
    reference: updated.reference,
    txId: res.transaction.id,
    guest: { id: guestRow.id, phone: guestRow.phone },
  });

  return Response.json({ ok: true, status: updated.status, txId: res.transaction.id });
}
