import { z } from "zod";
import { prisma } from "@/lib/prisma";

const BodySchema = z.object({
  phone: z.string().min(7).max(20),
});

function makeOtp() {
  return Math.floor(100000 + Math.random() * 900000).toString();
}

export async function POST(req: Request) {
  const json = await req.json().catch(() => null);
  const parsed = BodySchema.safeParse(json);
  if (!parsed.success) return Response.json({ error: "INVALID_BODY" }, { status: 400 });

  const phone = parsed.data.phone.trim();
  const oneMinuteAgo = new Date(Date.now() - 60_000);
  const recent = await prisma.otpChallenge.findFirst({
    where: { phone, createdAt: { gte: oneMinuteAgo }, purpose: "LOGIN" },
    orderBy: { createdAt: "desc" },
  });
  if (recent) return Response.json({ error: "RATE_LIMIT" }, { status: 429 });

  const code = makeOtp();
  const expiresAt = new Date(Date.now() + 5 * 60_000);
  const ch = await prisma.otpChallenge.create({
    data: { phone, purpose: "LOGIN", code, expiresAt },
  });

  const devOtp = process.env.NODE_ENV !== "production" ? code : undefined;
  return Response.json({ ok: true, challengeId: ch.id, expiresAt: ch.expiresAt, devOtp });
}
