import Link from "next/link";
import { TopNav } from "@/components/nav";
import { Button, Card, CardBody, CardHeader, Container, Input, Label } from "@/components/ui";
import { prisma } from "@/lib/prisma";
import { requireRole } from "@/lib/requireRole";
import { runSettlementBatchAction } from "@/app/admin/actions";
import { fromMinor } from "@/lib/money";

export default async function AdminSettlementPage() {
  await requireRole(["ADMIN"]);

  const batches = await prisma.settlementBatch.findMany({
    orderBy: { runAt: "desc" },
    take: 20,
    include: { lines: true },
  });

  return (
    <>
      <TopNav />
      <Container>
        <div className="grid gap-6 lg:grid-cols-2">
          <Card>
            <CardHeader title="Settlement" subtitle="Batch reconciliation for partners" />
            <CardBody>
              <div className="text-sm text-zinc-600">
                Produces a batch summary and CSV export (stub). In production, this becomes the source-of-truth report for
                Orange Money / Smega / My Zaka / banks.
              </div>
              <form action={runSettlementBatchAction} className="mt-4 grid gap-3">
                <div>
                  <Label>Run note (optional)</Label>
                  <Input name="note" placeholder="Nightly settlement" />
                </div>
                <Button type="submit">Run settlement batch</Button>
              </form>
            </CardBody>
          </Card>

          <Card>
            <CardHeader title="Recent batches" subtitle="Last 20" />
            <CardBody>
              {batches.length === 0 ? (
                <div className="text-sm text-zinc-600">No batches yet.</div>
              ) : (
                <div className="space-y-4">
                  {batches.map((b) => (
                    <div key={b.id} className="rounded-xl border border-zinc-200 p-3">
                      <div className="flex items-center justify-between">
                        <div>
                          <div className="text-sm font-medium">{new Date(b.runAt).toLocaleString()}</div>
                          <div className="text-xs text-zinc-600">{b.note ?? "—"}</div>
                        </div>
                        <Link className="text-sm underline" href={`/api/admin/settlement/${b.id}/csv`}>
                          Export CSV
                        </Link>
                      </div>
                      <div className="mt-2 grid gap-2">
                        {b.lines.map((l) => (
                          <div key={l.id} className="flex items-center justify-between text-xs">
                            <div className="text-zinc-700">
                              {l.provider} · {l.direction} · {l.count} tx
                            </div>
                            <div className="font-medium">P {fromMinor(l.amount)}</div>
                          </div>
                        ))}
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </CardBody>
          </Card>
        </div>
      </Container>
    </>
  );
}
