"use server";

import { prisma } from "@/lib/prisma";
import { requireRole } from "@/lib/requireRole";

export async function setUserRoleAction(_: any, formData: FormData) {
  await requireRole(["ADMIN"]);
  const userId = String(formData.get("userId") ?? "");
  const role = String(formData.get("role") ?? "");
  if (!userId || !["CONSUMER", "MERCHANT", "ADMIN"].includes(role)) return { ok: false };
  await prisma.user.update({ where: { id: userId }, data: { role: role as any } });
  return { ok: true };
}

export async function approveKycAction(_: any, formData: FormData) {
  await requireRole(["ADMIN"]);
  const userId = String(formData.get("userId") ?? "");
  if (!userId) return { ok: false };
  await prisma.user.update({ where: { id: userId }, data: { kycLevel: "FULL" } });
  await prisma.kycProfile.upsert({
    where: { userId },
    update: { status: "APPROVED", reviewedAt: new Date() },
    create: { userId, status: "APPROVED", submittedAt: new Date(), reviewedAt: new Date() },
  });
  return { ok: true };
}

export async function rejectKycAction(_: any, formData: FormData) {
  await requireRole(["ADMIN"]);
  const userId = String(formData.get("userId") ?? "");
  const reason = String(formData.get("reason") ?? "").slice(0, 200);
  if (!userId) return { ok: false };
  await prisma.user.update({ where: { id: userId }, data: { kycLevel: "BASIC" } });
  await prisma.kycProfile.upsert({
    where: { userId },
    update: { status: "REJECTED", reviewedAt: new Date(), rejectionReason: reason },
    create: { userId, status: "REJECTED", submittedAt: new Date(), reviewedAt: new Date(), rejectionReason: reason },
  });
  return { ok: true };
}

export async function resolveDisputeAction(_: any, formData: FormData) {
  await requireRole(["ADMIN"]);
  const disputeId = String(formData.get("disputeId") ?? "");
  const action = String(formData.get("action") ?? "");
  const note = String(formData.get("note") ?? "").slice(0, 500);
  if (!disputeId || !["RESOLVED", "DISMISSED", "IN_REVIEW"].includes(action)) return { ok: false };
  await prisma.dispute.update({
    where: { id: disputeId },
    data: {
      status: action as any,
      adminNote: note || null,
      resolvedAt: action === "RESOLVED" || action === "DISMISSED" ? new Date() : null,
    },
  });
  return { ok: true };
}

export async function runSettlementBatchAction(_: any, formData: FormData) {
  await requireRole(["ADMIN"]);
  const note = String(formData.get("note") ?? "").slice(0, 200);

  const last = await prisma.settlementBatch.findFirst({ orderBy: { runAt: "desc" } });
  const since = last?.runAt ?? new Date(Date.now() - 24 * 60 * 60_000);

  const transfers = await prisma.externalTransfer.findMany({
    where: { status: "SUCCEEDED", createdAt: { gt: since } },
    orderBy: { createdAt: "asc" },
  });

  const byKey = new Map<string, { provider: string; direction: string; amount: bigint; count: number }>();
  for (const t of transfers) {
    const key = `${t.provider}:${t.type}`;
    const cur = byKey.get(key) ?? { provider: t.provider, direction: t.type, amount: 0n, count: 0 };
    cur.amount += t.amount;
    cur.count += 1;
    byKey.set(key, cur);
  }

  const totals = {
    since: since.toISOString(),
    through: new Date().toISOString(),
    transferCount: transfers.length,
    totalAmountMinor: transfers.reduce((a, b) => a + b.amount, 0n).toString(),
    currency: "BWP",
  };

  const batch = await prisma.settlementBatch.create({
    data: {
      note: note || null,
      totals,
    },
  });

  const lines = Array.from(byKey.values());
  if (lines.length) {
    await prisma.settlementLine.createMany({
      data: lines.map((l) => ({
        batchId: batch.id,
        provider: l.provider,
        direction: l.direction,
        amount: l.amount,
        currency: "BWP",
        count: l.count,
      })),
    });
  }

  return { ok: true, batchId: batch.id, totals };
}
