"use server";

import { z } from "zod";
import { requireSession } from "@/lib/session";
import { creditWallet, debitWallet, transferLefaToLefa } from "@/lib/ledger";
import { prisma } from "@/lib/prisma";
import { getProvider } from "@/connectors/providers";

const SendSchema = z.object({
  toEmail: z.string().email(),
  amount: z.coerce.number().positive(),
  description: z.string().max(140).optional(),
});

export async function sendMoneyAction(_: any, formData: FormData) {
  const session = await requireSession();

  const parsed = SendSchema.safeParse({
    toEmail: formData.get("toEmail"),
    amount: formData.get("amount"),
    description: formData.get("description"),
  });

  if (!parsed.success) {
    return { ok: false, error: "INVALID_INPUT" };
  }

  const amountMinor = BigInt(Math.round(parsed.data.amount * 100));

  try {
    const res = await transferLefaToLefa({
      fromUserId: session.uid,
      toEmail: parsed.data.toEmail,
      amountMinor,
      description: parsed.data.description,
    });

    return { ok: true, txId: res.transaction.id };
  } catch (e: any) {
    return { ok: false, error: e?.message ?? "FAILED" };
  }
}

const TopupSchema = z.object({
  amount: z.coerce.number().positive().max(10000),
});

export async function demoTopupAction(_: any, formData: FormData) {
  const session = await requireSession();

  const parsed = TopupSchema.safeParse({ amount: formData.get("amount") });
  if (!parsed.success) return { ok: false, error: "INVALID_INPUT" };

  const amountMinor = BigInt(Math.round(parsed.data.amount * 100));
  try {
    const res = await creditWallet({ userId: session.uid, amountMinor, description: "Demo top-up" });
    return { ok: true, txId: res.transaction.id };
  } catch (e: any) {
    return { ok: false, error: e?.message ?? "FAILED" };
  }
}

const ExternalFundingSchema = z.object({
  direction: z.enum(["TOPUP", "WITHDRAWAL"]),
  provider: z.enum(["ORANGE_MONEY", "SMEGA", "MY_ZAKA", "BANK"]),
  phoneOrAccount: z.string().min(5).max(64),
  amount: z.coerce.number().positive().max(100000),
});

export async function externalFundingAction(_: any, formData: FormData) {
  const session = await requireSession();
  const parsed = ExternalFundingSchema.safeParse({
    direction: formData.get("direction"),
    provider: formData.get("provider"),
    phoneOrAccount: formData.get("phoneOrAccount"),
    amount: formData.get("amount"),
  });
  if (!parsed.success) return { ok: false, error: "INVALID_INPUT" };

  const amountMinor = BigInt(Math.round(parsed.data.amount * 100));
  const provider = getProvider(parsed.data.provider as any);
  const reference = `EXT_${Date.now()}_${Math.random().toString(36).slice(2, 8)}`.toUpperCase();

  const wallet = await prisma.walletAccount.upsert({
    where: { userId: session.uid },
    update: {},
    create: { userId: session.uid, balance: 0n, currency: "BWP" },
  });

  const ext = await prisma.externalTransfer.create({
    data: {
      type: parsed.data.direction,
      provider: provider.name,
      walletId: wallet.id,
      amount: amountMinor,
      currency: "BWP",
      reference,
      meta: { phoneOrAccount: parsed.data.phoneOrAccount },
    },
  });

  // Stub adapter returns SUCCEEDED immediately. Replace with real async callbacks.
  const res = await provider.initiate({
    type: parsed.data.direction,
    phoneOrAccount: parsed.data.phoneOrAccount,
    amountMinor,
    currency: "BWP",
    reference,
  });

  await prisma.externalTransfer.update({
    where: { id: ext.id },
    data: {
      status: res.status,
      externalRef: res.externalRef,
      meta: { ...(ext.meta as any), ...(res.meta ?? {}) },
    },
  });

  try {
    if (parsed.data.direction === "TOPUP" && res.status === "SUCCEEDED") {
      const tx = await creditWallet({
        userId: session.uid,
        amountMinor,
        description: `${provider.displayName} top-up`,
      });
      return { ok: true, txId: tx.transaction.id, externalTransferId: ext.id };
    }
    if (parsed.data.direction === "WITHDRAWAL" && res.status === "SUCCEEDED") {
      const tx = await debitWallet({
        userId: session.uid,
        amountMinor,
        description: `${provider.displayName} withdrawal`,
      });
      return { ok: true, txId: tx.transaction.id, externalTransferId: ext.id };
    }
    return { ok: false, error: "PROVIDER_FAILED", externalTransferId: ext.id };
  } catch (e: any) {
    return { ok: false, error: e?.message ?? "FAILED", externalTransferId: ext.id };
  }
}
